import heapq
from typing import List, Dict, Tuple

from jpype import JInt
from regex import D


def dfa2adj_mat(dfa) -> List[List[int]]:
    """
    Converts a deterministic finite automaton (DFA) into an adjacency matrix representation.

    Args:
        dfa (object): The DFA object representing the automaton.

    Returns:
        list: A 2D adjacency matrix where the matrix element at row i and column j is 1 if there is a transition from state i to state j on input symbol j, and 0 otherwise.

    """

    states = dfa.getStates()
    alphabet = dfa.getInputAlphabet()

    # Initialize a 2D adjacency matrix with zeros
    adjacency_matrix = [[0 for _ in range(len(states))] for _ in range(len(states))]

    # Fill the matrix based on transitions
    for state in states:
        for input_symbol in alphabet:
            trans = dfa.getTransition(JInt(state), JInt(alphabet.getSymbolIndex(input_symbol)))
            successor = dfa.getIntSuccessor(trans)
            if successor != -1:  # Ensure there's a valid transition
                adjacency_matrix[state][successor] = 1

    return adjacency_matrix


def dijkstra(graph: List[List[int]], start: int) -> List[int]:
    """
    Dijkstra's algorithm to find the shortest path in a graph.

    Parameters:
    - graph: A list of lists representing the graph where each inner list corresponds to a node and contains integers representing edge weights to other nodes.
    - start: An integer representing the starting node in the graph.

    Returns:
    - A list of integers representing the distances from the starting node to all other nodes in the graph.
    """

    num_nodes = len(graph)
    distances = [float('inf')] * num_nodes
    distances[start] = 0
    priority_queue = [(0, start)]
    
    while priority_queue:
        current_distance, current_node = heapq.heappop(priority_queue)
        
        if current_distance > distances[current_node]:
            continue
        
        for neighbor, weight in enumerate(graph[current_node]):
            if neighbor != current_node and weight != 0:  # Ignore non-edges and self-loops
                distance = current_distance + weight
                
                if distance < distances[neighbor]:
                    distances[neighbor] = distance
                    heapq.heappush(priority_queue, (distance, neighbor))
    
    return [int(_) for _ in distances]


def learn_value_function_from_dfa(dfa) -> Dict[int, Dict[str, int]]:
    alphabet = dfa.getInputAlphabet()
    states = dfa.getStates()
    value_function = {}

    # Initialize value function and distance to acceptance
    inf = float('inf')
    size = dfa.size()
    accDists = [inf for _ in range(size)]

    for i in range(size):
        i = JInt(i)
        if dfa.isAccepting(i):
            accDists[i] = 0

    while True:
        stable = True
        for state in range(size):
            succMinDist = inf
            for sym in alphabet:
                trans = dfa.getTransition(JInt(state), JInt(alphabet.getSymbolIndex(sym)))
                succ = dfa.getIntSuccessor(trans)
                succMinDist = min(succMinDist, accDists[succ])
            if succMinDist == inf:
                continue
            succMinDist += 1
            if succMinDist < accDists[state]:
                accDists[state] = succMinDist
                stable = False
        if stable:
            break

    for state in states:
        value_function[state] = {}
        for input_symbol in alphabet:
            trans = dfa.getTransition(JInt(state), JInt(alphabet.getSymbolIndex(input_symbol)))
            succ = dfa.getIntSuccessor(trans)
            value_function[state][input_symbol] = -accDists[succ]

    return value_function


